<?php
require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/utils.php';

function h($s){ return htmlspecialchars($s ?? '', ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
function is_admin(){
    return isset($_SESSION['user']) && isset($_SESSION['user']['login']) && in_array($_SESSION['user']['login'], ADMIN_LOGINS, true);
}

// Handle comment submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add_comment') {
    if (!isset($_SESSION['user'])) {
        header('Location: ' . AUTH_LOGIN_URL . '&redirect_uri=' . urlencode(SITE_URL . $_SERVER['REQUEST_URI']));
        exit;
    }
    
    $post_id = intval($_POST['post_id']);
    $content = trim($_POST['content']);
    
    if (!empty($content) && $post_id > 0) {
        $stmt = $db->prepare("INSERT INTO comments (post_id, user_id, user_login, user_name, content, created_at) VALUES (?, ?, ?, ?, ?, NOW())");
        $stmt->execute([
            $post_id,
            $_SESSION['user']['id'] ?? null,
            $_SESSION['user']['login'],
            $_SESSION['user']['name'] ?? $_SESSION['user']['login'],
            $content
        ]);
    }
    
    header('Location: ' . $_SERVER['REQUEST_URI']);
    exit;
}

// Handle comment editing
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'edit_comment') {
    if (!isset($_SESSION['user'])) {
        header('Location: ' . AUTH_LOGIN_URL . '&redirect_uri=' . urlencode(SITE_URL . $_SERVER['REQUEST_URI']));
        exit;
    }
    
    $comment_id = intval($_POST['comment_id']);
    $content = trim($_POST['content']);
    
    if (!empty($content) && $comment_id > 0) {
        // Check if user owns the comment or is admin
        $stmt = $db->prepare("SELECT user_login FROM comments WHERE id = ?");
        $stmt->execute([$comment_id]);
        $comment = $stmt->fetch();
        
        if ($comment && ($comment['user_login'] === $_SESSION['user']['login'] || is_admin())) {
            $stmt = $db->prepare("UPDATE comments SET content = ?, updated_at = NOW() WHERE id = ?");
            $stmt->execute([$content, $comment_id]);
        }
    }
    
    header('Location: ' . $_SERVER['REQUEST_URI']);
    exit;
}

// Handle comment deletion
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete_comment') {
    if (!isset($_SESSION['user'])) {
        header('Location: ' . AUTH_LOGIN_URL . '&redirect_uri=' . urlencode(SITE_URL . $_SERVER['REQUEST_URI']));
        exit;
    }
    
    $comment_id = intval($_POST['comment_id']);
    
    if ($comment_id > 0) {
        // Check if user owns the comment or is admin
        $stmt = $db->prepare("SELECT user_login FROM comments WHERE id = ?");
        $stmt->execute([$comment_id]);
        $comment = $stmt->fetch();
        
        if ($comment && ($comment['user_login'] === $_SESSION['user']['login'] || is_admin())) {
            $stmt = $db->prepare("DELETE FROM comments WHERE id = ?");
            $stmt->execute([$comment_id]);
        }
    }
    
    header('Location: ' . $_SERVER['REQUEST_URI']);
    exit;
}

// Start session with persistent settings if not already started
if (session_status() == PHP_SESSION_NONE) {
    session_set_cookie_params([
        'lifetime' => 315360000, // 10 years in seconds
        'path' => '/',
        'domain' => '',
        'secure' => true,
        'httponly' => true,
        'samesite' => 'Lax'
    ]);
    session_start();
}
?>
<!DOCTYPE html>
<html lang="ru">
<head>
<meta charset="utf-8">
<title>Архив.ДуДу2</title>
<style>
/* Centered container styles */
.container-wrapper {
  display: flex;
  justify-content: center;
  align-items: flex-start;
  min-height: 100vh;
  padding: 20px;
  background-color: #f0f0f0;
  font-family: Verdana, Arial, sans-serif;
}

.main-container {
  width: 100%;
  max-width: 1200px;
  background-color: white;
  border: 1px solid #ccc;
  box-shadow: 0 0 10px rgba(0,0,0,0.1);
}

.header {
  background: linear-gradient(to bottom, #0099cc, #007790, #005a70);
  color: white;
  padding: 10px 15px;
  display: flex;
  justify-content: space-between;
  align-items: center;
  border-bottom: 1px solid #ccc;
}

.header-title {
  font-size: 18px;
  font-weight: bold;
}

.user-info {
  font-size: 12px;
}

.user-info a {
  color: white;
  text-decoration: none;
}

.user-info a:hover {
  text-decoration: underline;
}

.content-wrapper {
  display: flex;
  padding: 0;
}

.sidebar {
  width: 220px;
  background-color: #f5f5f5;
  border-right: 1px solid #ddd;
  padding: 15px;
}

.sidebar-section {
  margin-bottom: 20px;
}

.sidebar-title {
  background: linear-gradient(to bottom, #0099cc, #007790, #005a70);
  color: white;
  padding: 5px 8px;
  font-size: 12px;
  font-weight: bold;
  margin-bottom: 10px;
  border: 1px solid #005a70;
}

.sidebar-link {
  display: block;
  color: #0000cc;
  text-decoration: none;
  font-size: 12px;
  margin-bottom: 5px;
  padding: 2px 0;
}

.sidebar-link:hover {
  text-decoration: underline;
}

.search-form input[type="text"] {
  width: 100%;
  padding: 3px;
  font-size: 12px;
  margin-bottom: 5px;
  border: 1px solid #999;
}

.search-form input[type="submit"] {
  font-size: 12px;
  padding: 2px 8px;
  background-color: #007790;
  color: white;
  border: 1px solid #005a70;
  cursor: pointer;
}

.search-form input[type="submit"]:hover {
  background-color: #005a70;
}

.categories-list {
  font-size: 12px;
}

.categories-list a {
  color: #0000cc;
  text-decoration: none;
}

.categories-list a:hover {
  text-decoration: underline;
}

.main-content {
  flex: 1;
  padding: 15px;
}

.post-title {
  font-size: 18px;
  font-weight: bold;
  font-family: Tahoma, Verdana, Arial;
  margin-bottom: 10px;
}

.post-meta {
  font-size: 12px;
  font-family: Verdana;
  margin-bottom: 10px;
  padding-bottom: 10px;
  border-bottom: 1px solid #ccc;
}

.post-content {
  font-size: 12px;
  font-family: Verdana;
  line-height: 1.5;
}

.post-tags {
  font-size: 12px;
  font-family: Verdana;
  font-weight: bold;
  margin: 15px 0;
}

/* Table styling */
.content-table {
  width: 100%;
  border-collapse: collapse;
  margin-top: 15px;
}

.content-table th {
  background: linear-gradient(to bottom, #0099cc, #007790, #005a70);
  color: white;
  padding: 6px;
  text-align: left;
  font-weight: bold;
  font-size: 12px;
  font-family: Verdana;
}

.content-table td {
  padding: 6px;
  border-bottom: 1px solid #eee;
  font-size: 12px;
  font-family: Verdana;
}

.content-table tr:hover {
  background-color: #f0f0f0;
}

.message {
  padding: 10px;
  margin-bottom: 15px;
  border-radius: 3px;
  font-family: Verdana;
  font-size: 12px;
}

.message.success {
  background-color: #ddffdd;
  border: 1px solid #4CAF50;
}

.message.error {
  background-color: #ffdddd;
  border: 1px solid #f44336;
}

.footer {
  background-color: #f0f0f0;
  padding: 10px 15px;
  text-align: right;
  border-top: 1px solid #ddd;
  font-size: 11px;
  color: #666;
}


</style>

</head>
<body>
<div class="container-wrapper">
  <div class="main-container">
    <div class="header">
      <div class="header-title">Архив.ДуДу2</div>
      <div class="user-info">
        <?php if(isset($_SESSION['user'])): ?>
          Пользователь: <b><?php echo h($_SESSION['user']['name'] ?? $_SESSION['user']['login']); ?></b> |
          <?php if(is_admin()): ?><a href="admin.php">Админ панель</a> |<?php endif; ?>
          <a href="index.php?action=logout">Выход</a>
        <?php else: ?>
          <a href="<?php echo h(AUTH_LOGIN_URL . '&redirect_uri=' . urlencode(SITE_URL)); ?>">Войти</a>
        <?php endif; ?>
      </div>
    </div>

    <div class="content-wrapper">
      <div class="sidebar">
        <div class="sidebar-section">
          <div class="sidebar-title">Навигация</div>
          <a href="index.php" class="sidebar-link">Последние записи</a>
          <a href="browse.php" class="sidebar-link">Все категории</a>
          <?php if(is_admin()): ?><a href="admin.php" class="sidebar-link">Админ панель</a><?php endif; ?>
        </div>
        
        <div class="sidebar-section">
          <div class="sidebar-title">Поиск</div>
          <form action="search.php" method="get" class="search-form">
            <div style="display: flex;">
              <input type="text" name="q" size="15" style="flex: 1; margin-right: 5px;">
              <input type="submit" value="Найти" style="flex: 0 0 auto; padding: 3px 8px;">
            </div>
          </form>
        </div>
        
        <div class="sidebar-section">
          <div class="sidebar-title">Категории</div>
          <div class="categories-list">
            <?php
            echo generateCategoryNav($db);
            ?>
          </div>
        </div>
      </div>
      
      <div class="main-content">
<?php
$slug = isset($_GET['slug']) ? trim($_GET['slug']) : '';
$st = $db->prepare("SELECT p.*, c.name as catname, c.slug as cslug, u_login FROM posts p LEFT JOIN categories c ON c.id=p.category_id WHERE p.slug=?");
$st->execute([$slug]);
$post = $st->fetch();
if(!$post){
    echo '<font face=Verdana size=-1>Запись не найдена.</font>';
} else {
    echo '<div class="post-title">'.h($post['title']).'</div>';
    echo '<div class="post-meta">Категория: <a href="browse.php?cat='.h($post['cslug']).'">'.h($post['catname']).'</a> | Дата: '.h($post['created_at']);
    
    // Add edit link for administrators
    if(is_admin()){
        echo ' | <a href="admin.php?edit='.$post['id'].'">Редактировать</a>';
    }
    
    echo '</div>';
    echo '<div class="post-content">'. $post['content_html'] .'</div>';

    // Use prepared statement for tags query
    $tags = $db->prepare("SELECT t.name FROM tags t INNER JOIN post_tags pt ON pt.tag_id=t.id WHERE pt.post_id=? ORDER BY t.name");
    $tags->execute([$post['id']]);
    $t = $tags->fetchAll(PDO::FETCH_COLUMN);
    if($t){
        echo '<div class="post-tags">Теги: '.h(implode(', ', $t)).'</div>';
    }

    // Try to get attachments with sort_order, fallback to uploaded_at if sort_order doesn't exist
    try {
        $att = $db->prepare("SELECT * FROM attachments WHERE post_id=? ORDER BY sort_order, uploaded_at");
        $att->execute([$post['id']]);
    } catch (PDOException $e) {
        // If sort_order column doesn't exist, fallback to uploaded_at
        if (strpos($e->getMessage(), 'sort_order') !== false) {
            $att = $db->prepare("SELECT * FROM attachments WHERE post_id=? ORDER BY uploaded_at");
            $att->execute([$post['id']]);
        } else {
            throw $e;
        }
    }
    $a = $att->fetchAll();
    if($a){
        echo '<table class="content-table">';
        echo '<tr><th>Файл</th><th>Размер</th><th>Скачать</th></tr>';
        foreach($a as $f){
            echo '<tr><td>'.h($f['original_name']).'</td>';
            echo '<td>'.number_format($f['size']/1024, 1, ',', ' ').' КБ</td>';
            echo '<td><a href="uploads/'.h($f['filename']).'" download>Скачать</a></td></tr>';
        }
        echo '</table>';
    }
    
    // Display comments
    echo '<div class="comments-section">';
    echo '<h3 style="font-size: 16px; margin: 20px 0 10px 0;">Комментарии:</h3>';
    
    // Fetch comments for this post
    $comments_stmt = $db->prepare("SELECT * FROM comments WHERE post_id = ? ORDER BY created_at ASC");
    $comments_stmt->execute([$post['id']]);
    $comments = $comments_stmt->fetchAll();
    
    if ($comments) {
        foreach ($comments as $comment) {
            echo '<div class="comment" style="border: 1px solid #ddd; padding: 10px; margin-bottom: 10px; background-color: #f9f9f9;">';
            echo '<div class="comment-meta" style="font-size: 11px; color: #666; margin-bottom: 5px;">';
            echo '<strong>' . h($comment['user_name']) . '</strong> | ' . h($comment['created_at']);
            if (!empty($comment['updated_at'])) {
                echo ' (отредактировано: ' . h($comment['updated_at']) . ')';
            }
            // Edit/Delete options for comment owners or admins
            if (isset($_SESSION['user']) && ($comment['user_login'] === $_SESSION['user']['login'] || is_admin())) {
                echo ' | <a href="#" onclick="editComment(' . $comment['id'] . '); return false;">Редактировать</a> | ';
                echo '<a href="#" onclick="deleteComment(' . $comment['id'] . '); return false;">Удалить</a>';
            }
            echo '</div>';
            echo '<div class="comment-content" id="comment-content-' . $comment['id'] . '" style="font-size: 12px;">' . nl2br(h($comment['content'])) . '</div>';
            
            // Edit form (hidden by default)
            echo '<form method="post" id="edit-comment-form-' . $comment['id'] . '" style="display: none; margin-top: 10px;">';
            echo '<input type="hidden" name="action" value="edit_comment">';
            echo '<input type="hidden" name="comment_id" value="' . $comment['id'] . '">';
            echo '<textarea name="content" rows="3" style="width: 100%; font-family: Verdana; font-size: 12px;">' . h($comment['content']) . '</textarea>';
            echo '<div style="margin-top: 5px;">';
            echo '<input type="submit" value="Сохранить" style="font-size: 12px; padding: 2px 8px; background-color: #007790; color: white; border: 1px solid #005a70; cursor: pointer;">';
            echo '<input type="button" value="Отмена" onclick="cancelEdit(' . $comment['id'] . ');" style="font-size: 12px; padding: 2px 8px; margin-left: 5px; background-color: #007790; color: white; border: 1px solid #005a70; cursor: pointer;">';
            echo '</div>';
            echo '</form>';
            
            echo '</div>';
        }
    } else {
        echo '<p style="font-size: 12px; color: #666;">Пока нет комментариев.</p>';
    }
    
    // Comment form for logged-in users
    if (isset($_SESSION['user'])) {
        echo '<div class="add-comment" style="margin-top: 20px; padding-top: 10px; border-top: 1px solid #ddd;">';
        echo '<h4 style="font-size: 14px; margin-bottom: 10px;">Добавить комментарий:</h4>';
        echo '<form method="post">';
        echo '<input type="hidden" name="action" value="add_comment">';
        echo '<input type="hidden" name="post_id" value="' . $post['id'] . '">';
        echo '<textarea name="content" rows="4" style="width: 100%; font-family: Verdana; font-size: 12px;" placeholder="Введите ваш комментарий..."></textarea>';
        echo '<div style="margin-top: 5px;">';
        echo '<input type="submit" value="Добавить комментарий" style="font-size: 12px; padding: 2px 8px; background-color: #007790; color: white; border: 1px solid #005a70; cursor: pointer;">';
        echo '</div>';
        echo '</form>';
        echo '</div>';
    } else {
        echo '<div class="login-to-comment" style="margin-top: 20px; padding-top: 10px; border-top: 1px solid #ddd;">';
        echo '<p style="font-size: 12px; color: #666;">Чтобы оставить комментарий, <a href="' . h(AUTH_LOGIN_URL . '&redirect_uri=' . urlencode(SITE_URL . $_SERVER['REQUEST_URI'])) . '" style="color: #0000cc;">войдите</a>.</p>';
        echo '</div>';
    }
    
    echo '</div>';
    
    // Add JavaScript for comment editing/deletion
    echo '<script>
    function editComment(commentId) {
        document.getElementById("comment-content-" + commentId).style.display = "none";
        document.getElementById("edit-comment-form-" + commentId).style.display = "block";
    }
    
    function cancelEdit(commentId) {
        document.getElementById("comment-content-" + commentId).style.display = "block";
        document.getElementById("edit-comment-form-" + commentId).style.display = "none";
    }
    
    function deleteComment(commentId) {
        if (confirm("Вы уверены, что хотите удалить этот комментарий?")) {
            var form = document.createElement("form");
            form.method = "post";
            form.innerHTML = \'<input type=\"hidden\" name=\"action\" value=\"delete_comment\">\' +
                             \'<input type=\"hidden\" name=\"comment_id\" value=\"\' + commentId + \'\">\';
            document.body.appendChild(form);
            form.submit();
        }
    }
    </script>';
}


?>
      </div>
    </div>
    
    <div class="footer">
      © <?php echo date('Y'); ?> Архив.ДуДу2
    </div>
  </div>
</div>
</body>
</html>